#!/usr/bin/perl

# regions file generator script
# February 15 2008  v02 CRADU

use POSIX qw(floor ceil);

# usage regionsgen.pl <config.input> <input.scl> 
# all inputs are mandatory

#version control
my $REGIONSGEN_VERSION = "v02";

#input parameters
my $config_file_name 	= $ARGV[0]; #config input file name
my $scl_file_name 	= $ARGV[1]; #input scl file name

#db for input config data
#for macro definition stage
my $macro_height; #macro max height
my %macro_width; #keep macros width and height
#for group definition
my %group_block; 	#keeps the blocks from the group
my %group_macro; 	#keeps the groups macros types and their number
my $group_height; 	#max height fot the group
my $group_width; 	#calculated group width based on his content
my $group_space_h;	#store the group spacing
my $group_space_v;
#for cluster definition
my $cluster_p_width;
my $cluster_p_height;
my $cluster_space_h = 0;	#store cluster spacing
my $cluster_space_v = 0;
#for property definition
my %property_db;
my $temp_prop_list;

#check input parameters
if( !defined($config_file_name) || !defined($scl_file_name) ) {
  print "Usage: regionsgen.pl <config.input> <input.scl>\n";
  exit(0);
}

#open global log file 
open (EFILE,">regionsgen.log") || die "unable to create regionsgen.log file\n";

#get bbox of the placement grid defined in .scl file
my @scl_bbox = &process_scl( $scl_file_name );
if ( @scl_bbox != 4 ) {#expecting ( lx, ly, hx, hy )
  my_exit();
}else{#debug only : print bbox
  print "bbox ( @scl_bbox )\n";
}

#parse input config file
unless ( &process_input_config($config_file_name) == 1 ){
  my_exit();#errors encountered
}

#debug only
&print_config_data();

#generate output.regions file
unless ( &generate_output_regions() == 1 ) {
  my_exit();#errors encountered while generating output.regions file
}

#close global error log file
close EFILE;

#end program
exit(0);


###################################################################################
#  FUNCTIONS
###################################################################################


####################################
# process input config file
# usage : process_input_config <input file name>
# return 0 if errors occured; 1 if OK
####################################
sub process_input_config
{
my $config_file_name = $_[0];

  unless ( open (CFILE, "<$config_file_name") ) {
    print       "unable to open $config_file_name file\n";
    print EFILE "unable to open $config_file_name file\n";
    return 0;
  }

  my $line;
  my @words;
  my $STATE = 0;
# possible states 
# 0 : waiting to encounter a paragraph
# 1 : reading macro definition paragraph
# 2 : reading group definition paragraph
# 3 : reading cluster definition paragraph
# 4 : reading property definition

  while ( defined($line = <CFILE>)) {
    @words = get_words_list($line);  
    if ( !@words || $words[0] eq "#" ) {#ignore empty or coments lines
      next;
    }

    if( $STATE == 0 ) { #waiting to enconter a paragraph
      if( $words[0] eq "Macros" ) { $STATE = 1; next; } #begin macro definition
      if( $words[0] eq "Group" ) 	{ $STATE = 2; next; } #begin group definition
      if( $words[0] eq "Cluster" ){ $STATE = 3; next; } #begin cluster definition
      if( $words[0] eq "Property" ) { $STATE = 4; next; } #begin property definition
      #none of the above and also not an empty line or comment ?? ERROR
      print       "unexpected .config file line \"$line\" \"$words[0]\"\n";
      print EFILE "unexpected .config file line $line\n";
      close CFILE;          
      return 0;
      next; #unnecessary ( for dummy code updates ) go to the next line
    }
    if( $STATE == 1 ) { #reading macro definitions
      if( $words[0] eq "End" ) { $STATE = 0; next; } #end of macro defition
      #expect : macro_name width height
      if( exists $macro_width{$words[0]} ) {#check if the macro was already defined
        print       "macro $words[0] defined twice\n";
        print EFILE "macro $words[0] defined twice\n";
        close CFILE;
        return 0;
      }
      if( @words < 3 ) { #check if there are atleast 3 elemets in this line
        print       "insuficient parameters in line : \"$line\"\n";
        print EFILE "insuficient parameters in line : \"$line\"\n";
        close CFILE;
        return 0;
      }
      $macro_width{$words[0]} = $words[1];#add macro width and height to the dbs
      $macro_height{$words[0]} = $words[2];
      next; # go to the next line
    }
    if( $STATE == 2 ) { #reading group definition
      if( $words[0] eq "End" ) { $STATE = 0; next; } #end of group definition
      #expect : Height <height>
      if( $words[0] eq "Height" ) { $group_height = $words[1]; next; }         
      #expect : Spacing <horizontaly> <veritcaly>
      if( $words[0] eq "Spacing" ) { $group_space_h = $words[1];
                                     $group_space_v = $words[2];
                                     next; }      
      #expect : macro_name number block_no base_col
      unless( exists $macro_width{$words[0]} ) {#check if the macro has a definition
        print       "macro $words[0] wasn't defined in macros definition\n";
        print EFILE "macro $words[0] wasn't defined in macros definitions\n";
        close CFILE;
        return 0;
      }
      if( @words < 4 ) { #check is there are at least 4 parameters in this line
        print       "insuficient parameters in line : \"$line\"\n";
        print EFILE "insuficient parameters in line : \"$line\"\n";
        close CFILE;
        return 0;
      }
      $group_macro{$words[0]} += $words[1];#add macro and number to the group DB  
      
      my $temp_block_element = { name     => $words[0] ,#add macro to the group blocks DB
                                 elem_no  => $words[1] ,
                                 col_atom => $words[3] };
      push @{ $group_block{$words[2]} } , $temp_block_element;
      
      next; # go to the next line
    }
    if( $STATE == 3 ) { #reading cluster definition
      if( $words[0] eq "End" ) { $STATE = 0; next; } #end of cluster definition
      #expect : Spacing <horizontaly> <veritcaly>
      if( $words[0] eq "Spacing" ) { $cluster_space_h = $words[1];
                                     $cluster_space_v = $words[2];
                                     next; }  
      #expect : width height
      if( @words < 2 ) { #check is there are at least 2 parameters in this line
        print       "insuficient parameters in line : \"$line\"\n";
        print EFILE "insuficient parameters in line : \"$line\"\n";
        close CFILE;
        return 0;
      }
      $cluster_p_width = $words[0];
      $cluster_p_height = $words[1];
      next; # go to the next line
    }
    if( $STATE == 4 ) { #reading property definition
      if( $words[0] eq "End" ) { $STATE = 0; next; } #end of property definition
      #expect : Name <property name>
      if( $words[0] eq "Name" ) {    
        if( exists $property_db{$words[1]} ) {#check if the property is listed twice
          print       "property $words[1] is listed twice\n";
          print EFILE "property $words[1] is listed twice\n";
          close CFILE;
          return 0;
        }
        my %temp_property; 
        $property_db{$words[1]} = \%temp_property;
        $temp_prop_list = \%temp_property;
        next;
      }
      #expect : level_name number
      if( exists ${$temp_prop_list}{$words[0]} ) {#check if the prop level is listed twice
        print       "prop level $words[0] is listed twice in group definition\n";
        print EFILE "prop level $words[0] is listed twice in group definition\n";
        close CFILE;
        return 0;
      }
      if( @words < 2 ) { #check is there are at least 2 parameters in this line
        print       "insuficient parameters in line : \"$line\"\n";
        print EFILE "insuficient parameters in line : \"$line\"\n";
        close CFILE;
        return 0;
      }
      ${$temp_prop_list}{$words[0]} = $words[1];#add prop level and number to the prop DB
      next; # go to the next line
    }

    #invalid state ????
    print       "invalid state \"$STATE\": code error\n";
    print EFILE "invalid state \"$STATE\": code error\n";
    close CFILE;
    return 0;
  }

  unless( $STATE == 0 ) {#check if the files ends waiting for a paragraph
    print       "invalid end of file state \"$STATE\": code error\n";
    print EFILE "invalid end of file state \"$STATE\": code error\n";
    close CFILE;
    return 0;
  }

  close CFILE;  
  return 1; #return succes
}#end processing input config file
  

#############################################
# print input config db ( debug purposes )
#############################################
sub print_config_data
{#print stored input data 
  print "macro definition\n";
  foreach (keys %macro_width) {
    print "M : \"$_\" , W : $macro_width{$_}, H : $macro_height{$_}\n";
  }
  
  print"\ngroup content\n";
  print "group height $group_height\n";
  foreach (keys %group_macro) {
    print "M :\"$_\" , Num : $group_macro{$_}\n";
  }
  
  print "group blocks\n";
  my $block_id;
  foreach $block_id (sort keys %group_block) {
    print "block_id $block_id\n";
    foreach (@{$group_block{$block_id}}) {
      print "macro_name $_->{name} element_no $_->{elem_no}\n";
    }
  }
  
  print"\ncluster dim : W $cluster_p_width , H : $cluster_p_height\n";
  
  my $prop;
  print "\nproperty content\n";
  foreach $prop (keys %property_db){
    print "\nprop $prop\n";
    foreach (keys %{$property_db{$prop}}){
      print "L: $_ Num : ${$property_db{$prop}}{$_}\n";
    }
  }
}#end printing debug info



#############################################
# generating output .regions file
# return 0 if errors occured ; 1 if OK
#############################################
sub generate_output_regions
{
  unless ( &create_regions_file() == 1 ){#create file and write header
    return 0;
  }
 
  #generate region file up to group level 
  $group_width = &generate_upto_group_regions();
  #debug info 
  print "group width = $group_width\n";

  #generate regions for above group level hierarchy ( cluster & top )
  &generate_above_group_regions();

  #close output file
  close RFILE;
  return 1;      
}

#############################################
# create output.regions file and write his header
# return 0 if error occured; 1 if OK
############################################# 
sub create_regions_file
{
  unless ( open (RFILE, ">output.regions") ) {
    print       "unable to create output.regions file\n";
    print EFILE "unable to create output.regions file\n";
    return 0;
  }

  #write .regions header
  print RFILE "eASIC regions 1.0\n";
  print RFILE "# Created\t: .regions script generator $REGIONSGEN_VERSION\n";
  print RFILE "# User\t\t:\n";
  
  return 1;
}     
############################################
# generate group&columns regions
# return group width based on his content
############################################
sub generate_upto_group_regions
{

  my %column_width; #column DB; keep height and width of a group column
  my %column_height;
  my %column_type;
  my %group_column; #stores the columns in all the blocks in the group
  my %blocks_in_group; #stores blocks from the group (name and width)
  
  my $col_element_num;
  my $col_number;
  my $macro_name;
  my $column_name;
  
  
  #process blocks detrmine number of columns and their order
  my $block_id;
  foreach $block_id (sort keys %group_block) {

    foreach (@{$group_block{$block_id}}) {
      $_->{column_name} = $_->{name} . "_column";
      $col_element_num = $group_height / $macro_height{ $_->{name} };
      $_->{col_number} = $_->{elem_no} / $col_element_num;
      $column_type{$_->{column_name}} = $_->{name};#add macro types that are accepted in this col
      $group_column{$_->{column_name}} = 1;  #add column to the col map
      $column_width{$_->{column_name}} = $macro_width{$_->{name}};#add column w and h info
      $column_height{$_->{column_name}} = $col_element_num * $macro_height{$_->{name}};           
    } 
  }
 
 
#write columns regions definition
  foreach $column_name (sort keys %group_column) {
    print RFILE "\n\n# $column_name area definition\n";
    print RFILE "PropArea\n";
    print RFILE "Name\t: $column_name\n";
    print RFILE "Width\t: $column_width{$column_name}\n";
    print RFILE "Height\t: $column_height{$column_name}\n";
    print RFILE "Property : type\t Values : $column_type{$column_name} \n";
    &write_color_props($column_name);
    print RFILE "# end prop area $column_name\n";
    print RFILE "EndPropArea\n";
  }
                                           
 
#write blocks area definitions 
 foreach $block_id (sort keys %group_block) {   
   print RFILE "\n\n# block_$block_id area definition\n";
   print RFILE "PropArea\n";
   print RFILE "Name\t: block_$block_id\n";
   &write_color_props("block_$block_id");
   my $block_width_count = 0;
   my $write_column_flag = 1;
   while ( $write_column_flag ) {
     $write_column_flag = 0;
     foreach (@{$group_block{$block_id}}) {
       my $index;
       for( $index = 0 ; $index < $_->{col_atom} ; $index++ ) {
         if( $_->{col_number} > 0 ){
           print RFILE "\tPropAreaInst : $_->{column_name} : $block_width_count : 0\n";
           $block_width_count += $column_width{$_->{column_name}};
           $_->{col_number}--;
           $write_column_flag = 1;
   }}}}
     
   print RFILE "# end prop area block_$block_id\n";
   print RFILE "EndPropArea\n";                                    
   
   $blocks_in_group{"block_$block_id"} = $block_width_count;
 } 

  #write group prop area
  print RFILE "\n\n# group area definition\n";
  print RFILE "PropArea\n";
  print RFILE "Name\t: group\n";
  &write_color_props("group");
  print RFILE "# list of instances\n";

  my $group_width_count = 0;
  foreach $block_name ( sort keys %blocks_in_group ) {#write instances of columns
    print RFILE "\tPropAreaInst : $block_name : $group_width_count : 0\n";
    $group_width_count += $blocks_in_group{$block_name};
  }
  print RFILE "# end prop area group\n";
  print RFILE "EndPropArea\n";        
  
  return $group_width_count;
}


########################################
# generate cluster & top level  regions
########################################
sub generate_above_group_regions
{  
  my $top_width = $scl_bbox[2] - $scl_bbox[0];
  my $top_height = $scl_bbox[3] - $scl_bbox[1];
  
  my $cluster_height = $cluster_p_height * $top_height;
  my $cluster_width = $cluster_p_width * $top_width;

  #fill cluster with group instances
  &fill_master_prop_area ( "cluster",$cluster_width,$cluster_height,"group",$group_width,$group_height,$group_space_h, $group_space_v);
  #fill top with cluster instances
  &fill_master_prop_area ( "top",$top_width,$top_height,"cluster",$cluster_width,$cluster_height,$cluster_space_h, $cluster_space_v);

  #instantiate top prop area in the lower left corner of the scl defined area
  print RFILE "\n# top chip instance\n";
  print RFILE "PropAreaInst : top : $scl_bbox[0] : $scl_bbox[1]\n";    
}


##################################################
# writes the "color" properties for a prop area 
# usage write_color_props ( <prop area name> )
##################################################
sub write_color_props
{
  my $prop_level = $_[0];
  unless (defined $prop_level) { return }

  my $prop;
  foreach $prop (keys %property_db){
    if( exists ${$property_db{$prop}}{$prop_level}){
      print RFILE "Property : $prop\t\tNumColors : ${$property_db{$prop}}{$prop_level}\n";
    }
  }                          
}



#######################################
# fills a master area with 
# as many as possible slave instances 
#######################################
sub fill_master_prop_area 
{
my ( $master_name, $master_width, $master_height, $slave_name, $slave_width, $slave_height, $slave_space_h, $slave_space_v) = @_;

  if( $master_height < $slave_height || $master_width < $slave_width ) {
    print 	"Invalid master/slave geometry\n";
    print EFILE "Invalid master/slave geometry\n";
    return ;
  }
  my $x_count = floor ($master_width / ( $slave_width + $slave_space_h ));
  my $y_count = floor ($master_height / ( $slave_height + $slave_space_v ));
  
  my $real_space_x = $master_width / $x_count ;
  my $real_space_y = $master_height / $y_count;
  
  print RFILE "\n\n# $master_name area definition\n";
  print RFILE "PropArea\n";
  print RFILE "Name\t: $master_name\n";
  &write_color_props($master_name);
  print RFILE "# list of instances\n";
          
  my ($i, $j, $x_coord, $y_coord);
  
  for( $i = 0 ; $i < $x_count; $i++ ){
    $x_coord = $i * $real_space_x;
    for ( $j = 0 ; $j < $y_count; $j++){
      $y_coord = $j * $real_space_y;
      printf RFILE "\tPropAreaInst : $slave_name : %.2f : %.2f\n", $x_coord, $y_coord ;
    }
  }
  
  print RFILE "# end prop area $master_name\n";
  print RFILE "EndPropArea\n";
}

############################################################
# scl file processing function
# input  : .scl file name
# output : .scl defined BBOX ( lx, ly, hx, hy )
############################################################
sub process_scl
{
my $scl_file_name = $_[0];
  
  unless ( defined $scl_file_name ) { return }
  
  unless ( open (SFILE,"<$scl_file_name") ){
    print 	"unable to open .scl file $scl_file_name\n";
    print EFILE "unable to open .scl file $scl_file_name\n";
    return ;    
  }


  my @RowDB;	#stores row y low value for each row
  my @RowDBstartX;#starting X coordinate of each row
  my @RowDBendX;	#ending X coordinate of each row. RowsDBs share the same index

  my $WINDOW_LX = 10000000000;
  my $WINDOW_LY = 10000000000;
  my $WINDOW_HX = -10000000000;
  my $WINDOW_HY = -10000000000;

  my $site_orient;
  my $site_width;
  my $num_processed_rows = 0;
  my $row_height;
  my $prev_row_height;
  my $line;
  while (defined($line = <SFILE>)) {
    @words = &get_words_list($line);
            
    if ( !defined $words[0] || $words[0] eq "#" || $words[0] eq "UCLA") {
      next;
    }
    if ($words[0] eq "NumRows") {
      $num_rows = $words[2];
      print "NumRows: $num_rows are defined\n";
      next;
    }
    if ($words[0] eq "CoreRow") {
      $line = <SFILE>;
      @words = &get_words_list($line);
      
      # Keyword: Coordinate
      if ($words[0] eq "Coordinate") {
        $row_y = $words[2];
      }
      else {
        print EFILE "ERROR: CoreRow Processing: Coordinate keyword not found\n";
        print       "ERROR: CoreRow Processing: Coordinate keyword not found\n";
        return ;
      }
      push(@RowDB, $row_y);
      
      # Keyword: Height
      $line = <SFILE>;
      @words = &get_words_list($line);
                                                                                                                                                                                                                                      
      if ($words[0] eq "Height") {
        $row_height = $words[2];
      }else {
        print EFILE "ERROR: CoreRow Processing: Height keyword not found\n";
        print       "ERROR: CoreRow Processing: Height keyword not found\n";
        return ;
      }
                                                          
      if( defined $prev_row_height && $prev_row_height != $row_height ) {
        print EFILE "ERROR: Row Height mismatch: $prev_row_height vs $row_height\n";
        print       "ERROR: Row Height mismatch: $prev_row_height vs $row_height\n";
        return ;
      }
      $prev_row_height = $row_height;
                                                                                                              
      # Keyword: Sitewidth
      $line = <SFILE>;
      @words = &get_words_list($line);
                                                                                                                                                      
      if ($words[0] eq "Sitewidth") {
        $site_width = $words[2];
      }else {
        print EFILE "ERROR: CoreRow Processing: Sitewidth keyword not found\n";
        print       "ERROR: CoreRow Processing: Sitewidth keyword not found\n";
        return ;
      }
                                                                              
      # Keyword: Sitespacing
      $line = <SFILE>;
      @words = &get_words_list($line);
    
      if ($words[0] eq "Sitespacing") {
        $site_width = $words[2];
      }else {
        print EFILE "ERROR: CoreRow Processing: Sitespacing keyword not found\n";
        print       "ERROR: CoreRow Processing: Sitespacing keyword not found\n";
        return ;
      }
                                                                                    
      # Keyword: Siteorient
      $line = <SFILE>;
      @words = &get_words_list($line);
                                                                                                                            
      if ($words[0] eq "Siteorient") {
        $site_orient = $words[2];
      }else {
        print EFILE "ERROR: CoreRow Processing: Siteorient keyword not found. $words[0] $words[1]\n";
        print       "ERROR: CoreRow Processing: Siteorient keyword not found. $words[0] $words[1]\n";
        return ;
      }
      # Keyword: Sitesymmetry
      $line = <SFILE>;
      @words = &get_words_list($line);
      
      if ($words[0] eq "Sitesymmetry") {
        $site_width = $words[2];
      }else {
        print EFILE "ERROR: CoreRow Processing: Sitesymmetry keyword not found. $word[0] $word[1]\n";
        print       "ERROR: CoreRow Processing: Sitesymmetry keyword not found. $word[0] $word[1]\n";
        return ;
      }
                                                                              
      # Keyword: SubrowOrigin
      $line = <SFILE>;
      @words = &get_words_list($line);
                                                                                                                      
      if ($words[0] eq "SubrowOrigin") {
        $row_x = $words[2];
        $row_num_sites = $words[5];
        push(@RowDBstartX, $row_x);
        push(@RowDBendX, $row_x + $row_num_sites);
      }else {
        print EFILE "ERROR: CoreRow Processing: SubrowOrigin keyword not found\n";
        print       "ERROR: CoreRow Processing: SubrowOrigin keyword not found\n";
        return ;
      }
 
      if ($WINDOW_LX > $row_x) {
        $WINDOW_LX = $row_x;
      }
      if ($WINDOW_HX < $row_x + $row_num_sites) {
        $WINDOW_HX = $row_x + $row_num_sites;
      }
      if ($WINDOW_LY > $row_y) {
        $WINDOW_LY = $row_y;
      }
      if ($WINDOW_HY < $row_y + $row_height) {
        $WINDOW_HY = $row_y + $row_height;
      }

      $num_processed_rows++;
    }
  }
  close(SFILE);
  print "Phase 0: Total $num_processed_rows rows are processed.\n";
  return ($WINDOW_LX, $WINDOW_LY, $WINDOW_HX, $WINDOW_HY);
}

##########################################
# process text file input line
# return list of words 
##########################################

sub get_words_list
{
chomp ( my $line = $_[0] );

  $line =~ s/^\s+//; # removes front/end white spaces
  $line =~ s/\s+$//;
  return split(/\s+/, $line);   
}           

##########################################
# exit routine
##########################################
sub my_exit
{
  print		"Abnormal condition happened. Exit\n";
  print EFILE	"Abnormal condition happened. Exit\n";
  exit(1);
}
                                                                                                                                                                                                                                          
                                                                                                                                                                                                                                            
                                                                                                                              

                                                                                                                                                                                                                                                                    